/*

BSD License

Copyright (c) 2002, Brent Simmons
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

*	Redistributions of source code must retain the above copyright notice,
	this list of conditions and the following disclaimer.
*	Redistributions in binary form must reproduce the above copyright notice,
	this list of conditions and the following disclaimer in the documentation
	and/or other materials provided with the distribution.
*	Neither the name of ranchero.com or Brent Simmons nor the names of its
	contributors may be used to endorse or promote products derived
	from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


*/

/*
	SearchEngines.m
	Huevos

	Created by Brent Simmons on Wed Apr 03 2002.
	Copyright (c) 2002 Brent Simmons. All rights reserved.
*/

#import "SearchEngines.h"


NSString *searchEnginesKey = @"Search Engines";
NSString *commandKeysKey = @"Command Keys";


@implementation SearchEngines


+ (void) sendNotification {
	
	/*
	Notify the world that the search engines list has changed.
	*/
	
	[[NSNotificationCenter defaultCenter]
		postNotificationName: @"SearchEnginesChanged" object: nil];
	} /*sendNotification*/


/*Creating, deleting, editing*/

+ (void) savePrefs {
	
	[[NSUserDefaults standardUserDefaults] synchronize];
	} /*savePrefs*/
	
	
+ (void) new: (NSString *) name url: (NSString *) url {
	
	NSMutableDictionary *updatedDictionary =
		[[[self searchEnginesDictionary] mutableCopy] autorelease];
	
	[updatedDictionary setObject: url forKey: name];
	
	[[NSUserDefaults standardUserDefaults] setObject: updatedDictionary forKey: searchEnginesKey];
	
	[self savePrefs];
	
	[self sendNotification];
	} /*new*/


+ (void) deleteNthSearchEngine: (int) n {
	
	NSMutableDictionary *updatedDictionary =
		[[[self searchEnginesDictionary] mutableCopy] autorelease];
	
	[updatedDictionary removeObjectForKey: [self nthSearchEngineName: n]];
	
	[[NSUserDefaults standardUserDefaults] setObject: updatedDictionary forKey: searchEnginesKey];
	
	[self savePrefs];

	[self sendNotification];
	} /*deleteNthSearchEngine*/


+ (void) setNthUrl: (NSString *) url n: (int) n {
	
	NSString *name = [self nthSearchEngineName: n];
	
	[self deleteNthSearchEngine: n];
	
	[self new: name url: url];
	} /*setNthUrl*/


+ (void) renameNthSearchEngine: (NSString *) name n: (int) n {
	
	NSString *url = [self nthSearchEngineUrl: n];
	NSString *key = [self nthSearchEngineCommandKey: n];
	
	[self deleteNthSearchEngine: n];
	
	[self new: name url: url];
	
	[self setCommandKeyForSearchEngine: key name: name];
	} /*renameNthSearchEngine*/


/*Getting info*/

+ (NSDictionary *) searchEnginesDictionary {
	
	return [[NSUserDefaults standardUserDefaults] objectForKey: searchEnginesKey];
	} /*searchEngineDictionary*/
	

+ (NSString *) urlFromName: (NSString *) name {
	
	return [[self searchEnginesDictionary] objectForKey: name];
	} /*urlFromName*/
	
	
+ (NSString *) nthSearchEngineUrl: (int) n {
	
	NSString *name = [self nthSearchEngineName: n];
		
	return [[self searchEnginesDictionary] objectForKey: name];
	} /*nthSearchEngineUrl*/
	
	
+ (NSString *) nthSearchEngineName: (int) n {
	
	return [[self arrayOfNames] objectAtIndex: n];
	} /*nthSearchEngine*/


+ (int) indexOfSearchEngine: (NSString *) name {
	
	/*
	Given a name, what is its index in the prefs table.
	*/
	
	return [[self arrayOfNames] indexOfObject: name];
	} /*indexOfSearchEngine*/
	
	
+ (NSArray *) arrayOfNames {

	/*
	Returns a sorted array of search engine names.
	*/
	
	NSArray *keys = [[self searchEnginesDictionary] allKeys];
	
	return [keys sortedArrayUsingSelector: @selector (caseInsensitiveCompare:)];
	} /*arrayOfNames*/


+ (int) count {
	
	return [[self searchEnginesDictionary] count];
	} /*count*/


/*Command keys*/

+ (NSDictionary *) commandKeysDictionary {
	
	NSDictionary *d = [[NSUserDefaults standardUserDefaults] objectForKey: commandKeysKey];
	
	if (d == nil)
		return [NSDictionary dictionary];
	
	return (d);
	} /*commandKeysDictionary*/
	

+ (NSString *) commandKeyForSearchEngine: (NSString *) name {
	
	return [[self commandKeysDictionary] objectForKey: name];	
	} /*commandKeyForSearchEngine*/
	

+ (void) setCommandKeyForSearchEngine: (NSString *) cmdKey name: (NSString *) name {
	
	NSMutableDictionary *updatedDictionary =
		[[[self commandKeysDictionary] mutableCopy] autorelease];
	
	if ((cmdKey == nil) || ([cmdKey isEqualToString: @""]))
		[updatedDictionary removeObjectForKey: name];
	
	else {
		
		NSString *fixedCommandKey = [[cmdKey lowercaseString] substringToIndex: 1];

		[updatedDictionary setObject: fixedCommandKey forKey: name];
		} /*else*/
	
	[[NSUserDefaults standardUserDefaults] setObject: updatedDictionary forKey: commandKeysKey];

	[self sendNotification];
	} /*setCommandKeyForSearchEngine*/


+ (void) setNthCommandKey: (NSString *) cmdKey n: (int) n {
	
	[self setCommandKeyForSearchEngine: cmdKey name: [self nthSearchEngineName: n]];	
	} /*setNthCommandKey*/
	
	
+ (NSString *) nthSearchEngineCommandKey: (int) n {
	
	NSString *name = [self nthSearchEngineName: n];

	return [self commandKeyForSearchEngine: name];
	} /*nthSearchEngineCommandKey*/


@end
